<?php
/**
 * Typography control class.
 *
 * @since  1.0.0
 * @access public
 */

class Makeup_Artist_Control_Typography extends WP_Customize_Control {

	/**
	 * The type of customize control being rendered.
	 *
	 * @since  1.0.0
	 * @access public
	 * @var    string
	 */
	public $type = 'makeup-artist-typography';

	/**
	 * Array 
	 *
	 * @since  1.0.0
	 * @access public
	 * @var    string
	 */
	public $l10n = array();

	/**
	 * Set up our control.
	 *
	 * @since  1.0.0
	 * @access public
	 * @param  object  $manager
	 * @param  string  $id
	 * @param  array   $args
	 * @return void
	 */
	public function __construct( $manager, $id, $args = array() ) {

		// Let the parent class do its thing.
		parent::__construct( $manager, $id, $args );

		// Make sure we have labels.
		$this->l10n = wp_parse_args(
			$this->l10n,
			array(
				'color'       => esc_html__( 'Font Color', 'makeup-artist' ),
				'family'      => esc_html__( 'Font Family', 'makeup-artist' ),
				'size'        => esc_html__( 'Font Size',   'makeup-artist' ),
				'weight'      => esc_html__( 'Font Weight', 'makeup-artist' ),
				'style'       => esc_html__( 'Font Style',  'makeup-artist' ),
				'line_height' => esc_html__( 'Line Height', 'makeup-artist' ),
				'letter_spacing' => esc_html__( 'Letter Spacing', 'makeup-artist' ),
			)
		);
	}

	/**
	 * Enqueue scripts/styles.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 */
	public function enqueue() {
		wp_enqueue_script( 'makeup-artist-ctypo-customize-controls' );
		wp_enqueue_style(  'makeup-artist-ctypo-customize-controls' );
	}

	/**
	 * Add custom parameters to pass to the JS via JSON.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 */
	public function to_json() {
		parent::to_json();

		// Loop through each of the settings and set up the data for it.
		foreach ( $this->settings as $setting_key => $setting_id ) {

			$this->json[ $setting_key ] = array(
				'link'  => $this->get_link( $setting_key ),
				'value' => $this->value( $setting_key ),
				'label' => isset( $this->l10n[ $setting_key ] ) ? $this->l10n[ $setting_key ] : ''
			);

			if ( 'family' === $setting_key )
				$this->json[ $setting_key ]['choices'] = $this->get_font_families();

			elseif ( 'weight' === $setting_key )
				$this->json[ $setting_key ]['choices'] = $this->get_font_weight_choices();

			elseif ( 'style' === $setting_key )
				$this->json[ $setting_key ]['choices'] = $this->get_font_style_choices();
		}
	}

	/**
	 * Underscore JS template to handle the control's output.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 */
	public function content_template() { ?>

		<# if ( data.label ) { #>
			<span class="customize-control-title">{{ data.label }}</span>
		<# } #>

		<# if ( data.description ) { #>
			<span class="description customize-control-description">{{{ data.description }}}</span>
		<# } #>

		<ul>

		<# if ( data.family && data.family.choices ) { #>

			<li class="typography-font-family">

				<# if ( data.family.label ) { #>
					<span class="customize-control-title">{{ data.family.label }}</span>
				<# } #>

				<select {{{ data.family.link }}}>

					<# _.each( data.family.choices, function( label, choice ) { #>
						<option value="{{ choice }}" <# if ( choice === data.family.value ) { #> selected="selected" <# } #>>{{ label }}</option>
					<# } ) #>

				</select>
			</li>
		<# } #>

		<# if ( data.weight && data.weight.choices ) { #>

			<li class="typography-font-weight">

				<# if ( data.weight.label ) { #>
					<span class="customize-control-title">{{ data.weight.label }}</span>
				<# } #>

				<select {{{ data.weight.link }}}>

					<# _.each( data.weight.choices, function( label, choice ) { #>

						<option value="{{ choice }}" <# if ( choice === data.weight.value ) { #> selected="selected" <# } #>>{{ label }}</option>

					<# } ) #>

				</select>
			</li>
		<# } #>

		<# if ( data.style && data.style.choices ) { #>

			<li class="typography-font-style">

				<# if ( data.style.label ) { #>
					<span class="customize-control-title">{{ data.style.label }}</span>
				<# } #>

				<select {{{ data.style.link }}}>

					<# _.each( data.style.choices, function( label, choice ) { #>

						<option value="{{ choice }}" <# if ( choice === data.style.value ) { #> selected="selected" <# } #>>{{ label }}</option>

					<# } ) #>

				</select>
			</li>
		<# } #>

		<# if ( data.size ) { #>

			<li class="typography-font-size">

				<# if ( data.size.label ) { #>
					<span class="customize-control-title">{{ data.size.label }} (px)</span>
				<# } #>

				<input type="number" min="1" {{{ data.size.link }}} value="{{ data.size.value }}" />

			</li>
		<# } #>

		<# if ( data.line_height ) { #>

			<li class="typography-line-height">

				<# if ( data.line_height.label ) { #>
					<span class="customize-control-title">{{ data.line_height.label }} (px)</span>
				<# } #>

				<input type="number" min="1" {{{ data.line_height.link }}} value="{{ data.line_height.value }}" />

			</li>
		<# } #>

		<# if ( data.letter_spacing ) { #>

			<li class="typography-letter-spacing">

				<# if ( data.letter_spacing.label ) { #>
					<span class="customize-control-title">{{ data.letter_spacing.label }} (px)</span>
				<# } #>

				<input type="number" min="1" {{{ data.letter_spacing.link }}} value="{{ data.letter_spacing.value }}" />

			</li>
		<# } #>

		</ul>
	<?php }

	/**
	 * Returns the available fonts.  Fonts should have available weights, styles, and subsets.
	 *
	 * @todo Integrate with Google fonts.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return array
	 */
	public function get_fonts() { return array(); }

	/**
	 * Returns the available font families.
	 *
	 * @todo Pull families from `get_fonts()`.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return array
	 */
	function get_font_families() {

		return array(
			'' => __( 'No Fonts', 'makeup-artist' ),
        'Abril Fatface' => __( 'Abril Fatface', 'makeup-artist' ),
        'Acme' => __( 'Acme', 'makeup-artist' ),
        'Anton' => __( 'Anton', 'makeup-artist' ),
        'Architects Daughter' => __( 'Architects Daughter', 'makeup-artist' ),
        'Arimo' => __( 'Arimo', 'makeup-artist' ),
        'Arsenal' => __( 'Arsenal', 'makeup-artist' ),
        'Arvo' => __( 'Arvo', 'makeup-artist' ),
        'Alegreya' => __( 'Alegreya', 'makeup-artist' ),
        'Alfa Slab One' => __( 'Alfa Slab One', 'makeup-artist' ),
        'Averia Serif Libre' => __( 'Averia Serif Libre', 'makeup-artist' ),
        'Bangers' => __( 'Bangers', 'makeup-artist' ),
        'Boogaloo' => __( 'Boogaloo', 'makeup-artist' ),
        'Bad Script' => __( 'Bad Script', 'makeup-artist' ),
        'Bitter' => __( 'Bitter', 'makeup-artist' ),
        'Bree Serif' => __( 'Bree Serif', 'makeup-artist' ),
        'BenchNine' => __( 'BenchNine', 'makeup-artist' ),
        'Cabin' => __( 'Cabin', 'makeup-artist' ),
        'Cardo' => __( 'Cardo', 'makeup-artist' ),
        'Courgette' => __( 'Courgette', 'makeup-artist' ),
        'Cherry Swash' => __( 'Cherry Swash', 'makeup-artist' ),
        'Cormorant Garamond' => __( 'Cormorant Garamond', 'makeup-artist' ),
        'Crimson Text' => __( 'Crimson Text', 'makeup-artist' ),
        'Cuprum' => __( 'Cuprum', 'makeup-artist' ),
        'Cookie' => __( 'Cookie', 'makeup-artist' ),
        'Chewy' => __( 'Chewy', 'makeup-artist' ),
        'Days One' => __( 'Days One', 'makeup-artist' ),
        'Dosis' => __( 'Dosis', 'makeup-artist' ),
        'Droid Sans' => __( 'Droid Sans', 'makeup-artist' ),
        'Economica' => __( 'Economica', 'makeup-artist' ),
        'Fredoka One' => __( 'Fredoka One', 'makeup-artist' ),
        'Fjalla One' => __( 'Fjalla One', 'makeup-artist' ),
        'Francois One' => __( 'Francois One', 'makeup-artist' ),
        'Frank Ruhl Libre' => __( 'Frank Ruhl Libre', 'makeup-artist' ),
        'Gloria Hallelujah' => __( 'Gloria Hallelujah', 'makeup-artist' ),
        'Great Vibes' => __( 'Great Vibes', 'makeup-artist' ),
        'Handlee' => __( 'Handlee', 'makeup-artist' ),
        'Hammersmith One' => __( 'Hammersmith One', 'makeup-artist' ),
        'Inconsolata' => __( 'Inconsolata', 'makeup-artist' ),
        'Indie Flower' => __( 'Indie Flower', 'makeup-artist' ),
        'IM Fell English SC' => __( 'IM Fell English SC', 'makeup-artist' ),
        'Julius Sans One' => __( 'Julius Sans One', 'makeup-artist' ),
        'Josefin Slab' => __( 'Josefin Slab', 'makeup-artist' ),
        'Josefin Sans' => __( 'Josefin Sans', 'makeup-artist' ),
        'Kanit' => __( 'Kanit', 'makeup-artist' ),
        'Lobster' => __( 'Lobster', 'makeup-artist' ),
        'Lato' => __( 'Lato', 'makeup-artist' ),
        'Lora' => __( 'Lora', 'makeup-artist' ),
        'Libre Baskerville' => __( 'Libre Baskerville', 'makeup-artist' ),
        'Lobster Two' => __( 'Lobster Two', 'makeup-artist' ),
        'Merriweather' => __( 'Merriweather', 'makeup-artist' ),
        'Monda' => __( 'Monda', 'makeup-artist' ),
        'Montserrat' => __( 'Montserrat', 'makeup-artist' ),
        'Muli' => __( 'Muli', 'makeup-artist' ),
        'Marck Script' => __( 'Marck Script', 'makeup-artist' ),
        'Noto Serif' => __( 'Noto Serif', 'makeup-artist' ),
        'Open Sans' => __( 'Open Sans', 'makeup-artist' ),
        'Overpass' => __( 'Overpass', 'makeup-artist' ),
        'Overpass Mono' => __( 'Overpass Mono', 'makeup-artist' ),
        'Oxygen' => __( 'Oxygen', 'makeup-artist' ),
        'Orbitron' => __( 'Orbitron', 'makeup-artist' ),
        'Patua One' => __( 'Patua One', 'makeup-artist' ),
        'Pacifico' => __( 'Pacifico', 'makeup-artist' ),
        'Padauk' => __( 'Padauk', 'makeup-artist' ),
        'Playball' => __( 'Playball', 'makeup-artist' ),
        'Playfair Display' => __( 'Playfair Display', 'makeup-artist' ),
        'PT Sans' => __( 'PT Sans', 'makeup-artist' ),
        'Philosopher' => __( 'Philosopher', 'makeup-artist' ),
        'Permanent Marker' => __( 'Permanent Marker', 'makeup-artist' ),
        'Poiret One' => __( 'Poiret One', 'makeup-artist' ),
        'Quicksand' => __( 'Quicksand', 'makeup-artist' ),
        'Quattrocento Sans' => __( 'Quattrocento Sans', 'makeup-artist' ),
        'Raleway' => __( 'Raleway', 'makeup-artist' ),
        'Rubik' => __( 'Rubik', 'makeup-artist' ),
        'Rokkitt' => __( 'Rokkitt', 'makeup-artist' ),
        'Russo One' => __( 'Russo One', 'makeup-artist' ),
        'Righteous' => __( 'Righteous', 'makeup-artist' ),
        'Slabo' => __( 'Slabo', 'makeup-artist' ),
        'Source Sans Pro' => __( 'Source Sans Pro', 'makeup-artist' ),
        'Shadows Into Light Two' => __( 'Shadows Into Light Two', 'makeup-artist'),
        'Shadows Into Light' => __( 'Shadows Into Light', 'makeup-artist' ),
        'Sacramento' => __( 'Sacramento', 'makeup-artist' ),
        'Shrikhand' => __( 'Shrikhand', 'makeup-artist' ),
        'Tangerine' => __( 'Tangerine', 'makeup-artist' ),
        'Ubuntu' => __( 'Ubuntu', 'makeup-artist' ),
        'VT323' => __( 'VT323', 'makeup-artist' ),
        'Varela Round' => __( 'Varela Round', 'makeup-artist' ),
        'Vampiro One' => __( 'Vampiro One', 'makeup-artist' ),
        'Vollkorn' => __( 'Vollkorn', 'makeup-artist' ),
        'Volkhov' => __( 'Volkhov', 'makeup-artist' ),
        'Yanone Kaffeesatz' => __( 'Yanone Kaffeesatz', 'makeup-artist' )
		);
	}

	/**
	 * Returns the available font weights.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return array
	 */
	public function get_font_weight_choices() {

		return array(
			'' => esc_html__( 'No Fonts weight', 'makeup-artist' ),
			'100' => esc_html__( 'Thin',       'makeup-artist' ),
			'300' => esc_html__( 'Light',      'makeup-artist' ),
			'400' => esc_html__( 'Normal',     'makeup-artist' ),
			'500' => esc_html__( 'Medium',     'makeup-artist' ),
			'700' => esc_html__( 'Bold',       'makeup-artist' ),
			'900' => esc_html__( 'Ultra Bold', 'makeup-artist' ),
		);
	}

	/**
	 * Returns the available font styles.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return array
	 */
	public function get_font_style_choices() {

		return array(
			'' => esc_html__( 'No Fonts Style', 'makeup-artist' ),
			'normal'  => esc_html__( 'Normal', 'makeup-artist' ),
			'italic'  => esc_html__( 'Italic', 'makeup-artist' ),
			'oblique' => esc_html__( 'Oblique', 'makeup-artist' )
		);
	}
}
